package topdown.concrete_operator;

import topdown.data_structures.*;
import topdown.term.Term;
import topdown.term.Variable;

import java.util.*;


public class ConcreteJoin implements ConcreteOperator {

    private List<ConcreteOperator> operators;
    private List<List<Term>> inputV;
    private List<Term> outputV;
    private ArrayList<Tuple> currentRow;

    public ConcreteJoin(
            List<ConcreteOperator> operators, List<List<Term>> inputV, List<Term> outputV
    ) {
        this.operators = operators;
        this.inputV = inputV;
        this.outputV = outputV;

        // currentRow represents one element of cartesian product
        // Ordered set of tuples t_1, ..., t_n from input operators O_1, ..., O_n respectively
        this.currentRow = new ArrayList<>();
        getNextCurrentRow();
        getNextMatching();
    }

    @Override
    public Tuple next() {
        if (currentRow.isEmpty()) return null;

        // from currentRow construct tuple
        // (first create a substitution based on the inputV vars
        // and then make selection based on outputV)
        Tuple result = new Tuple();
        HashMap<Variable, Term> subst = new HashMap<>();
        for (int i = 0; i < operators.size(); i++) {
            Tuple t = currentRow.get(i);
            if (t != null) {
                for (int j = 0; j < t.size(); j++) {
                    subst.put((Variable) inputV.get(i).get(j), t.get(j));
                }
            }
        }
        for (Term term : outputV) {
            result.add(term.substitute(subst));
        }

        getNextCurrentRow();
        getNextMatching();
        return result;
    }

    @Override
    public void reset() {
        for (ConcreteOperator op : operators) {
            op.reset();
        }
        currentRow.clear();
        getNextCurrentRow();
        getNextMatching();
    }

    @Override
    public ConcreteOperator instance() {
        ConcreteJoin result = new ConcreteJoin(
                operators, inputV, outputV
        );
        result.reset();

        return result;
    }

    /**
     * Emplace next element of cartesian product of all input operators into currentRow.
     * If there is no such element, empty the currentRow
     * */
    private void getNextCurrentRow() {
        if (currentRow.isEmpty()) {
            for (ConcreteOperator op : operators) {
                Tuple next = op.next();
                if (next == null) {
                    currentRow.clear();
                    return;
                }
                currentRow.add(next);
            }
            return;
        }

        for (int i = operators.size() - 1; i >= 0; i--) {
            Tuple next = operators.get(i).next();
            if (next != null) {
                currentRow.set(i, next);
                for (int j = i + 1; j < operators.size(); j++) {
                    operators.get(j).reset();
                    Tuple first = operators.get(j).next();
                    if (first == null) {
                        currentRow.clear();
                        return;
                    }
                    currentRow.set(j, first);
                }
                return;
            }
        }

        currentRow.clear();
    }

    /**
     * Skip elements of cartesian product, until the currentRow matches input vector
     * constraints (equality of variables specified in inputV for each operator)
     * */
    private void getNextMatching() {
        while (!currentRow.isEmpty()) {
            HashMap<Variable, Term> subst = new HashMap<>();
            boolean matched = true;

            for (int k = 0; k < operators.size(); k++) {
                List<Term> subgoal = inputV.get(k);

                for (int j = 0; j < subgoal.size(); j++) {
                    if (!Term.match(subgoal.get(j), currentRow.get(k).get(j), subst)) {
                        matched = false;
                        break;
                    }
                }

                if (!matched) break;
            }

            if (matched) return;
            getNextCurrentRow();
        }
        currentRow = new ArrayList<>();
    }
}
